/*:
 * @plugindesc 階層的な選択肢表示プラグイン
 * @target MZ
 * @help
 * このプラグインを使用すると、選択肢を階層的に表示することができます。
 *
 * プラグインコマンド:
 *  - StartNestedChoices: 階層的選択肢を開始
 *  - SetChoices: 階層ごとの選択肢を設定
 *  - CloseNestedChoices: 階層的選択肢を終了
 *
 * @command StartNestedChoices
 * @text 階層的選択肢を開始
 * @desc 階層選択肢の開始点を設定します。
 *
 * @command SetChoices
 * @text 選択肢を設定
 * @desc 特定の階層の選択肢を設定します。
 *
 * @arg choices
 * @type struct<Choice>[]
 * @text 選択肢
 * @desc この階層で表示する選択肢を設定します。
 *
 * @command CloseNestedChoices
 * @text 階層的選択肢を終了
 * @desc 階層選択肢を終了します。
 */

/*~struct~Choice:
 * @param name
 * @type string
 * @text 選択肢名
 * @desc この選択肢に表示される名前です。
 *
 * @param nextChoices
 * @type struct<Choice>[]
 * @text 次の階層の選択肢
 * @desc この選択肢を選んだ場合に表示される次の階層の選択肢。
 */

(() => {
    const pluginName = "NestedChoices";

    function debugLog(...args) {
        console.log("[NestedChoices Debug]", ...args);
    }

    PluginManager.registerCommand(pluginName, "StartNestedChoices", () => {
        debugLog("Nested choices started");
    });

    PluginManager.registerCommand(pluginName, "SetChoices", args => {
        debugLog("SetChoices command received with args:", args);
        const choices = JSON.parse(args.choices).map(c => (typeof c === "string" ? JSON.parse(c) : c));
        debugLog("Parsed choices:", choices);
        showNestedChoices(choices);
    });

    PluginManager.registerCommand(pluginName, "CloseNestedChoices", () => {
        debugLog("Nested choices closed");
        $gameMessage.clear();
    });

    // 最初の選択肢と次の選択肢を管理するウィンドウ
    function showNestedChoices(choices) {
        if (!choices || choices.length === 0) {
            debugLog("No choices to display.");
            return;
        }

        const choiceNames = choices.map(choice => choice.name);
        debugLog("Displaying choices:", choiceNames);

        // メッセージウィンドウはそのままに、別ウィンドウで選択肢を表示
        const choiceWindow = new Window_ChoiceList(0, 300); // 新しいウィンドウ位置調整
        choiceWindow.width = Graphics.boxWidth; // ウィンドウ幅
        choiceWindow.height = 180; // ウィンドウ高さ
        SceneManager._scene.addChild(choiceWindow);

        // setChoicesで選択肢を設定
        choiceWindow.setChoices(choiceNames, 0, -1);  // setChoicesを使用して選択肢を設定

        // 選択肢が選ばれたときの処理
        choiceWindow.setHandler('ok', index => {
            const selectedChoice = choices[index];
            debugLog("Selected choice:", selectedChoice);

            const nextChoices = selectedChoice.nextChoices 
                ? selectedChoice.nextChoices.map(c => (typeof c === "string" ? JSON.parse(c) : c))
                : null;

            if (nextChoices && nextChoices.length > 0) {
                debugLog("Next choices found:", nextChoices);

                // 次の選択肢ウィンドウを表示
                setTimeout(() => {
                    showNestedChoices(nextChoices);
                }, 10);
            } else {
                debugLog("No further choices.");
                choiceWindow.close();
            }
        });

        // キャンセルボタンの設定
        choiceWindow.setHandler('cancel', () => {
            debugLog("Choice window closed.");
            choiceWindow.close();
        });
    }
})();
